; Project:		TTPSU v1.3
; Description:	Turntable PSU and speed control
; Author:		Johnny Norre
;				DuNo Electronics I/S
;				Kettrup All 37
;				9200 Aalborg SV
;				DK-Denmark
; Support:		johnny@norre.dk
;
; (C)2006-2013 by DuNo Electronics I/S
;
;    Date		Version		Description
; 08.02.2006	  0.0		Project started
; 20.02.2006	  1.0		Initial version completed
; 11.03.2006	  1.1		Ramp speed on start up and on speed change.
;							HiVolt clock now changes between 10 secs. and 25 secs.
;							due to extra time required by ramping.
; 20.10.2009	  1.2		HiVolt is now applied both during start-up and during
;							a speed change (due to an error in v1.1 it was not
;							applied on start-up).
;							Hi-voltage display change from 5/15 sec to 10/25 sec.
; 10.05.2011	  1.21beta	Sent beta version to a builder of the project, who 
;							needed the new "High voltage" handling (coming in v1.3).
; 06.06.2011	  1.22beta	Sent beta version to a builder of the project. This 
;							version fixes a start-up race problem.
; 20.06.2011	  1.23beta	Added option for HD44780 display. BETA - not tested on
;							actual HD44780 yet!!!
; 20.03.2012	  1.24beta	Corrected adjustment display so it displays correct
;							value from -99.9 to +99.9 (but not beyond this).
; 21.03.2012	  1.24sv    Special version for Enrico - always turns at 33 rpm 
;							for 5 secs. after START (no matter if 33 or 45 rpm).
;							Define SV_ENRICO for this version.
; 03.01.2013	  1.3	    Hi-voltage now 10 sec, 25 sec or always.
;							Ability to compile for 16F883 as well as 16F870.


;#define SV_ENRICO	; Enrico special version. Always turn at 33 rpm for 5 seconds
					; after pressing START.

; main.asm
; --------
; The code has the following main functions:
; Entry
;	Configure inputs, setup interrupts
; InterruptServiceRoutine
;	Handles the TIMER1 interrupt and reprograms it.
; MainLoop
;	Polls for keyboard input and prepares next DAC output value
;	if needed.

; Select include file according to selected processor
#ifdef __16F870
	#include "p16F870.inc"
#endif
#ifdef __16F883
	#include "p16F883.inc"
#endif

	radix	dec

; *****************************
; ***						***
; ***  CONFIGURATION WORD   ***
; ***						***
; *****************************

#ifdef __16F870
	__config	_CP_OFF & _DEBUG_OFF & _WRT_ENABLE_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _PWRTE_ON & _WDT_OFF & _HS_OSC
#endif
#ifdef __16F883
	__config	0x2007,_FCMEN_OFF  & _IESO_ON & _CP_OFF & _DEBUG_ON & _PWRTE_OFF & _CPD_OFF & _LVP_OFF & _BOR_ON & _PWRTE_ON & _WDT_OFF & _HS_OSC & _MCLRE_ON
	__config	0x2008,_WRT_OFF & _BOR40V
#endif


; *****************************
; ***						***
; ***        DEFINES        ***
; ***						***
; *****************************

; Defines for conditional compiling.
; NOTE: If OUTPUT_50HZ is defined, the clock speed (XTAL) must be
;		20MHz and if OUTPUT_60HZ is defined the clock speed must
;		be 12MHz!

	; Uncomment one of the OUTPUT_XXXX defines
	;#define	OUTPUT_50HZ		; Turntable is a 50Hz model
	;#define	OUTPUT_60HZ		; Turntable is a 60Hz model
	#define	OVERCLOCK_60HZ	; Use with care: Assumes PIC overclocked to 24MHz for 60Hz output operation
	; Uncomment one of the defines below
	#define KS0066U		; Display uses KS0066U control chip
	;#define HD44780	; Display uses HD44780 control chip - BETA - NOT TESTED ON HD44780 DISPLAY!!!

	; Sanity checking of defines
	#ifdef OUTPUT_50HZ
		#define USE_50HZ_TABLES
		#ifdef OVERCLOCK_60HZ
			error	"Can't use OVERCLOCK_60HZ for 50Hz output operation!"
		#endif
		#ifdef OUTPUT_60HZ
			error	"Can't use OUTPUT_50HZ and OUTPUT_60HZ simultaniously."
		#else
			messg	"*** 50Hz output - use 20MHz XTAL  ***"
		#endif
	#else
		#ifdef OVERCLOCK_60HZ
			#define USE_50HZ_TABLES
			messg	"*** Overclocked 60Hz output - use 24MHz XTAL  ***"
		#else
			messg	"*** 60Hz output - use 12MHz XTAL  ***"
			#undefine USE_50HZ_TABLES
		#endif
	#endif


; *****************************
; ***						***
; ***       CONSTANTS       ***
; ***						***
; *****************************

; NOTE: For MIN_PERIOD and MAX_PERIOD, only the high byte is tested!
MIN_PERIOD	= 255	; Minimum period allowed. For 33.3 rpm pulley 
					; this gives a max. speed of 130 rpm. 
MAX_PERIOD	= 4096	; Maximum period allowed. For 45 rpm pulley
					; this gives a min. speed of 11 rpm.
					; Note: Ramping is started from this value.

LCD_RS		=	0	; PORTC bitno for RS line on display
LCD_E		=	1	; PORTC bitno for E line on display

BTN_1_MASK	=	0x01	; PORTA bitmask for key 1 (+/33 PULLEY)
BTN_2_MASK	=	0x02	; PORTA bitmask for key 2 (-/45 PULLEY)
BTN_3_MASK	=	0x04	; PORTA bitmask for key 3 (STOP/MENU)
BTN_1_BIT	= 	0		; Bit no. for key 1
BTN_2_BIT	= 	1		; Bit no. for key 2
BTN_3_BIT	= 	2		; Bit no. for key 3

VOLTAGE_CTRL=	3		; Bit no. for high voltage output.

BUTTON_DEBOUNCE_COUNT	= 50	; The no of ms. the button has to be pressed or released for the ButtonState to change.
BUTTON_LONG_COUNT		= 10	; The no of BUTTON_DEBOUNCE_COUNT periods that has to pass for a long press to appear.

RUNNING_BIT			= 7			; The bit no. in SpeedMode register that indicates if the turntable is running or stopped
SPEEDMODE_RUNNING	= 0x80		; See "SpeedMode" register for description of SPEEDMODE_XXXX constants
SPEEDMODE_16RPM		= 0
SPEEDMODE_33RPM		= 1
SPEEDMODE_45RPM		= 2
SPEEDMODE_78RPM		= 3

MODE_NORMAL		= 0		; See "Mode" register for description of MODE_XXXX constants
MODE_WELCOME	= 1
MODE_MENU		= 2
MODE_PULLEY33	= 3
MODE_PULLEY45	= 4
MODE_VOLTTOGGLE = 5
MODE_ADJUST		= 6
MODE_SPEEDS_DEC	= 7
MODE_SPEEDS_INC	= 8

; CLOCK_PERIOD - the period needed for 1 ms. to pass.
#ifdef OUTPUT_50HZ
#ifdef OVERCLOCK_60HZ
CLOCK_PERIOD = 6000	; The period needed for 1 ms. to pass for 24MHz X-tal
#else
CLOCK_PERIOD = 5000	; The period needed for 1 ms. to pass for 20MHz X-tal
#endif
#else
CLOCK_PERIOD = 3000	; The period needed for 1 ms. to pass for 12MHz X-tal
#endif


; *****************************
; ***						***
; ***        MACROS         ***
; ***						***
; *****************************
; LcdClear
;
; Clear display, and return to home address
LcdClear	macro
	clrf	LcdPosition
	movlw	0x01			; CLEAR
	call	LcdCtrlOut		; 82us-1.64ms completion time, wait for 2 ms. to be sure
	call	WaitFor1Ms
	call	WaitFor1Ms		
	endm

LcdHome	macro
	clrf	LcdPosition
	movlw	0x02			; RETURN HOME
	call	LcdCtrlOut		; 40us-1.6ms completion time, wait for 2 ms. to be sure
	call	WaitFor1Ms
	call	WaitFor1Ms		
	endm


; *****************************
; ***						***
; ***    CODE AREA START    ***
; ***						***
; *****************************

	org 	0
	goto	Entry

	org		4


; *****************************
; ***						***
; ***  INTERRUPT FUNCTION   ***
; ***						***
; *****************************

; BEGIN InterruptServiceRoutine
; 
; InterruptServiceRoutine
; -----------------------
; Handles CCP1 interrupts. 
;	Keeps track on for how many periods the DAC value must be
;	set on PORTB.
;	When the period has expired, reprograms PORTB with the new 
;	DAC value, sets up first new timer value,
;	and tells the main loop to fetch next DAC value.

InterruptServiceRoutine:
	; Save context registers
	movwf	W_Temp			
	swapf	STATUS,W
	movwf	STATUS_Temp
	clrf	STATUS			; Bank 0

IsrRamp:
	movf	RampDirection,F	; Test if we need to ramp the timer period
	btfsc	STATUS,Z
	goto	IsrDac
	decf	RampDivider,F	; Test if speed should be ramped
	btfss	STATUS,Z
	goto	IsrDac
	movlw	1				; Set up ffast ramp
	btfss	RampDirection,7	; Test if fast ramp or slow ramp
	addlw	9				; Slow ramp = 10 (1+9)
	movwf	RampDivider		; Next ramp occurs in 4 interrupts
	btfsc	RampDirection,1	; Are we ramping down?
	goto	IsrRampDown
IsrRampUp:
	incf	SpeedL,F
	btfsc	STATUS,Z
	incf	SpeedH,F
	goto	IsrRampTest
IsrRampDown:
	movf	SpeedL,F
	btfsc	STATUS,Z
	decf	SpeedH,F
	decf	SpeedL,F
IsrRampTest:
	; Reprogram timer
	bcf		T1CON,TMR1ON	; Temporary stop TIMER1
	movf	SpeedL,W		; Low byte of current speed
	movwf	CCPR1L
	movf	SpeedH,W		; High byte of current speed
	movwf	CCPR1H
	bsf		T1CON,TMR1ON	; Restart TIMER1
	movf	SpeedH,W		; Test if we are finished ramping
	subwf	SpeedHTmp,W
	btfss	STATUS,Z		
	goto	IsrDac			
	movf	SpeedL,W
	subwf	SpeedLTmp,W
	btfss	STATUS,Z
	goto	IsrDac
	clrf	RampDirection	; We are finished ramping!

IsrDac:
	; Test if the DAC period (the number of times the same
	; DAC value must be outputted) is finished.
	decfsz	DacPeriod,F		; Does DacPeriod become zero?
	goto	IsrExit 		; No, exit ISR

IsrNewDacValue:
	; Write out the new DAC value and set up the new DAC period.
	; Note:	The DAC period is actually written a little delayed
	;		with respect to writing the timer, but this have no
	;		influence on the actual period, as all DAC periods
	;		will have the exact same delay.
	movf	DacValue,W		
	movwf	PORTB			; Output DAC value to PORTB
	movf	DacNextPeriod,W
	movwf	DacPeriod		; Store new DAC period
	
	; Get the next period to use after the current period expires and 
	; the next value to use.
	btfsc	SpeedMode,RUNNING_BIT
	goto	GetNewDacPeriod
	; The turntable is in STOP mode - just output 32 = approximate NULL value.
	movlw	1
	movwf	DacNextPeriod
	movlw	32
	movwf	DacValue
	goto	IsrExit			; Reenable interrupts and exit
GetNewDacPeriod:
	movf	DacDirection,W
	addwf	DacValue,F		; Increment or decrement DacValue to the next value to be outputted.
	btfss	STATUS,Z		; If the DacValue becomes 0, change direction
	goto	GetNewDacPeriod2
	movlw	1				; Count up
	movwf	DacDirection
GetNewDacPeriod2:
	movf	DacValue,W
	xorlw	63
	btfss	STATUS,Z		; If the DacValue becomes 63, change direction
	goto	GetNewDacPeriod3
	movlw	255				; Count down
	movwf	DacDirection
GetNewDacPeriod3:
	movf	PCLATH,W		; Store PCLATH
	movwf	PCLATH_Temp
	movf	DacValue,W
	call	DacTable		; Get next period from DAC table
	movwf	DacNextPeriod
	movf	PCLATH_Temp,W	; Restore PCLATH
	movwf	PCLATH	
	movf	DacNextPeriod,F	; Test if DacNextPeriod is zero
	btfsc	STATUS,Z
	goto	GetNewDacPeriod ; If so, try next value in table

IsrExit:
	; Handle ClockCounter
	movf	SpeedL,W		; Note: SpeedL/SpeedH
	subwf	ClockCounterL,F
	movf	SpeedH,W
	btfss	STATUS,C
	addlw	1				; If the borrow flag was set, add 1 to the high value
	subwf	ClockCounterH,F
	btfsc	STATUS,C
	goto	IsrExit2
	; Increment KeyClock 
	incf	KeyClock,F
	; Add CLOCK_PERIOD  to the ClockTimerL/H
	movlw	low(CLOCK_PERIOD)
	addwf	ClockCounterL,F
	movlw	high(CLOCK_PERIOD)
	btfsc	STATUS,C
	addlw	1
	addwf	ClockCounterH,F

	; Handle HiVoltClock
	btfsc	HiVolt,1		; Test if we need to handle high-voltage
	goto	IsrExit2		; If HiVolt.1 is set, then always high voltage
	movf	HiVoltClockH,F	; Test if Zero - then we do not need to count anymore
	btfsc	STATUS,Z
	goto	IsrExit2		; No need to count anymore
	incf	HiVoltClockL,F
	btfsc	STATUS,Z
	incf	HiVoltClockH,F
	movf	HiVoltClockH,F	; Test if HiVoltTime has expiered
	btfss	STATUS,Z
	goto	IsrExit2
	; Set PORTA<3> as input changing to low voltage output
	errorlevel -302
	bsf		STATUS,RP0		; Bank 1
	movlw	0xFF
	movwf	TRISA
	errorlevel +302
IsrExit2:
#ifdef SV_ENRICO
	; Handle SV-version
	btfsc	SavedSpeedActive,1	
	goto	IsrExit3		
	movf	SavedSMClockH,F	
	btfsc	STATUS,Z
	goto	IsrExit3		
	incf	SavedSMClockL,F
	btfsc	STATUS,Z
	incf	SavedSMClockH,F
	movf	SavedSMClockH,F	
	btfss	STATUS,Z
	goto	IsrExit3
	; 
	movlw	2				
	movwf	SavedSpeedActive
IsrExit3:
#endif
	; Reenable CCP1 interrupts
	clrf	STATUS			; bank 0
	bcf		PIR1,CCP1IF		; Clear the compare interrupt
	; Restore context registers
	swapf	STATUS_Temp,W
	movwf	STATUS
	swapf	W_Temp,F
	swapf	W_Temp,W
	; Return from ISR
	retfie	
InterruptServiceRoutineEnd
; END InterruptServiceRoutine


; Include sine wave generation tables
#ifdef USE_50HZ_TABLES
DacTable:
		#include "Table100.inc"
#else
DacTable:
		#include "Table50.inc"
#endif	


; *****************************
; ***						***
; ***      ENTRY POINT      ***
; ***						***
; *****************************

; BEGIN Entry
; 
; Entry
; -----
; Entry point for code.
;	Disables ADC module 
;	Configures PORTA<0:2> as digital inputs (for keys)
;	Configures PORTA<3> as digital inputs (for reduced speed)
;	Configures PORTB<0:5> as digital outputs (for DAC)
;	Configures PORTC<0:1,4:7> as digital outputs for display
;	Configures PORTC<2> as digital output (reserved by CCP1)
;	All other port bits are set as inputs
;	Outputs 32d to PORTB - set null level for DAC
;	Initializes variables
;	Read configuration from EEPROM
;	Initializes interrupts
;	Initializes TIMER1
;	Initializes CCP1	
;	Initialize display (display start up)
;	Initialize display to 4 bit operation
;	Set display mode
;	Write startup message
;	Starts CCP1 and TIMER1
;	Hands over control to the main loop
Entry:
	#ifdef __16F883
	errorlevel -302			; Do not report "Not in Bank 0" warning
	BANKSEL	ANSEL
	clrf	ANSEL
	BANKSEL	ANSELH
	clrf	ANSELH
	errorlevel +302			; Enable "Not in bank 0" waring
	BANKSEL	PORTA
	#endif

	clrf	STATUS		
	clrf	PORTA

    ; Disable ADC
	bsf		STATUS,RP0		; Bank 1
	errorlevel -302			; Do not report "Not in Bank 0" warning
	movlw	0x06	
	movwf	ADCON1			; All digital 

	; Configure PORTA (Key input)
	movlw	0xFF
	movwf	TRISA			; PORTA<0:7> as inputs
	; Configure PORTB (DAC output)
	movlw	0xC0
	movwf	TRISB			; PORTB<0:5> as outputs -  <6:7> as inputs.

	; Configure PORTC (Display output)
	movlw	0x0C
	movwf	TRISC			; PORTC<0:1,4:7> as outputs, <2:3> as inputs.
	errorlevel +302			; Enable "Not in bank 0" waring
	clrf	STATUS			; Bank 0

	; Set PORTB to DAC value 32 (null level)
	movlw	32
	movwf	PORTB

	; Initialize variables
	movlw	32				
	movwf	DacValue		; Set initial DacValue to output
	movlw	SPEEDMODE_33RPM ; Default RPM = 33.3, not running
	movwf	SpeedMode
	clrf	Speeds
	clrf	EEPROM_Update	; Do not update EEPROM
	movlw	1				; Set initial length of DacValue output
	movwf	DacPeriod
	movwf	DacNextPeriod
	movwf	DacDirection	; Set the direction of traversal of the dac table
	movlw	low(CLOCK_PERIOD)	; Setup counter for 1ms. pause
	movwf	ClockCounterL
	movlw	high(CLOCK_PERIOD)
	movwf	ClockCounterH
	clrf	ButtonPending	; No buttons are being debounced
	clrf	ButtonCmd		; No button commands pending
	movlw	0x07			; Mark all buttons as up
	movwf	ButtonStatus	
	clrf	LcdPosition		; Next character will go to position 0 on the LCD
	clrf	HiVoltClockL
	clrf	HiVoltClockH
	clrf	RampDirection	; Do not ramp
	movlw	1
	movwf	RampDivider		; Ramp on every interrupt

	; Read configuration from EEPROM
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movlw	eSpeeds			; Read "eSpeeds" setting
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	call	EEPromRead		; Read current speed (bank=0)
	movwf	Speeds			
	call	UpdateSpeeds
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movlw	eHiVolt			; Read "eHiVolt" setting
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	call	EEPromRead		; Read HiVolt setting (bank=0)
	movwf	HiVolt

	; Set up interrupts
	clrf	INTCON			; Disable all interrupts
	bsf		STATUS,RP0		; bank 1
	errorlevel -302		
	bsf		PIE1,CCP1IE		; Enable TIMER1 interrupt
	bcf		STATUS,RP0		; bank 0
	errorlevel +302		
	bsf		INTCON, PEIE	; Enable Peripheral interrupt

	; Initialize TIMER1
	; Note: The first interrupt will not occur at the correct time. This
	; 		has no effect however, as the program always is in STOP mode
	;		for the first period.
	clrf	T1CON
	clrf	TMR1L
	clrf	TMR1H

	; Initialize CCP1, start CCP1 and TIMER1
	call	ReadEEPROMSpeed	; Read adjusted speed from EEPROM for SpeedMode
	clrf	RampDirection
	movf	SpeedLTmp,W
	movwf	SpeedL
	movf	SpeedHTmp,W
	movwf	SpeedH
	call	UpdateSpeed		; Programs CCP1 and starts TIMER1
	movlw	0x0B			; Configure CCP as "Compare mode, trigger special event"
	movwf	CCP1CON			; Starts CCP1
	bsf		INTCON, GIE		; Enable interrupts (global)

#ifdef SV_ENRICO
	clrf	SavedSpeedMode
	clrf	SavedSpeedActive
#endif
	; Initialize display (REQUIRED start up sequence)
	movlw	50				; Wait 50 ms. to ensure proper startup of display
	call	WaitForMs
	movlw	0x30			; LCD "FUNCTION SET", 8-bit operation
	movwf	PORTC
	call	WaitFor1Ms
	bsf		PORTC,LCD_E		; Rise E line on display
	call	WaitFor1Ms		; Hold it high for 1 ms.
	bcf		PORTC,LCD_E		; Lower it again - command is buffered.
	movlw	5
	call	WaitForMs		; Wait for 5 ms. (Display requirement = 4.5 ms.)
	movlw	0x30			; LCD "FUNCTION SET", 8-bit operation
	movwf	PORTC
	call	WaitFor1Ms
	bsf		PORTC,LCD_E		; Rise E line on display
	call	WaitFor1Ms		; Hold it high for 1 ms.
	bcf		PORTC,LCD_E		; Lower it again - command is buffered.
	call	WaitFor1Ms		; Wait for 1 ms. (Display requirement = 100 us.)
	movlw	0x30			; LCD "FUNCTION SET", 8-bit operation
	movwf	PORTC
	call	WaitFor1Ms
	bsf		PORTC,LCD_E		; Rise E line on display
	call	WaitFor1Ms		; Hold it high for 1 ms.
	bcf		PORTC,LCD_E		; Lower it again - command is buffered.
	call	WaitFor1Ms		; Wait for 1 ms. (We do not read the BUSY flag, so just waste some time)		

	; Initialize display to 4 bit operation
	movlw	0x20			; LCD "FUNCTION SET", 4-bit operation
	movwf	PORTC
	call	WaitFor1Ms
	bsf		PORTC,LCD_E		; Rise E line on display
	call	WaitFor1Ms		; Hold it high for 1 ms.
	bcf		PORTC,LCD_E		; Lower it again - command is buffered.
	call	WaitFor1Ms		; Wait for 1 ms. (We do not read the BUSY flag, so just waste some time)		
	; Complete 4-bit initialisation
	movlw	0x28			; LCD "FUNCTION SET", 4-bit operation, 2 line mode, 5x7 font - 2 line mode, as 1 line mode does not work on most 16x1 displays!
	call	LcdCtrlOut		; Send control message to display
	call	WaitFor1Ms

	; Set display mode
	movlw	0x0C			; LCD "DISPLAY ON/OFF CONTROL" - Display on, no cursor, no blink
	call	LcdCtrlOut
	call	WaitFor1Ms
	movlw	0x01			; LCD "CLEAR DISPLAY"
	call	LcdCtrlOut
	call	WaitFor1Ms
	movlw	0x06			; LCD "ENTRY MODE SET", increment RAM, no shift	
	call	LcdCtrlOut
	call	WaitFor1Ms

	; Write startup message
	movlw	MODE_WELCOME
	movwf	Mode
	call	DisplayMode
	
	; Display "Stop"
	movlw	MODE_NORMAL
	movwf	Mode
	call	DisplayMode

	goto	MainLoop		; On purpose!
; END Entry		


; *****************************
; ***						***
; ***       MAINLOOP        ***
; ***						***
; *****************************

; MainLoop
; --------
; Main loop of the code.
;	Poll for keyboard input.
; 	Update display
MainLoop:
	; Test for key presses and key releases
	movlw	BTN_1_MASK
	call	TestButton
	movlw	BTN_2_MASK
	call	TestButton
	movlw	BTN_3_MASK
	call	TestButton
	; Handle button commands
	call	HandleButtonCmd
	;call	RA5toggle
	#ifdef SV_ENRICO
	btfss	SavedSpeedActive,1	; If bit 1 set
	goto	MainLoop
	clrf	SavedSpeedActive
	movf	SavedSpeedMode,W
	movwf	SpeedMode
	bsf		SpeedMode,RUNNING_BIT	; Start turntable
	call	DisplayMode
	call	UpdateRpm
	#endif
	goto	MainLoop

RA5toggle:
	errorlevel -302			; Do not report "Not in Bank 0" warning
	BANKSEL	TRISA
	movlw	0xDF
	movwf	TRISA
	errorlevel +302				
	BANKSEL PORTA
	movlw	0x20
	movwf	PORTA
	return

; ******************************
; ***						 ***
; *** BUTTON INPUT FUNCTIONS ***
; ***						 ***
; ******************************

; HandleButtonCmd
; ---------------
; Test if a keyboard press has resultet in a keyboard command
HandleButtonCmd:
	movf	ButtonCmd,F		; Test if a button command has fired
	btfsc	STATUS,Z
	return					; No pending button command
	btfsc	ButtonCmd,7		; Is it a long keypress?
	goto	HandleButtonCmdLongKeypress
	; Short keypress
	btfsc	ButtonCmd,BTN_1_BIT
	goto	HandleButtonCmd_Button1
	btfsc	ButtonCmd,BTN_2_BIT
	goto	HandleButtonCmd_Button2
	goto	HandleButtonCmd_Button3
HandleButtonCmdLongKeypress:
	btfsc	ButtonCmd,BTN_1_BIT
	goto	HandleButtonCmd_Button1Long
	btfsc	ButtonCmd,BTN_2_BIT
	goto	HandleButtonCmd_Button2Long
	goto	HandleButtonCmd_Button3Long

HandleButtonCmd_Button1:
	clrf	ButtonCmd
	; Check current mode
	; If MODE_NORMAL 	-> Decrement speed setting
	; If MODE_MENU		-> Speed=Speed-0.1
	movf	Mode,F			; Take advantage of the fact, that MODE_NORMAL = 0
	btfss	STATUS,Z
	goto	HandleButtonCmd_Button1_SpeedAdj
HandleButtonCmd_Button1_PrevRpm:
 	movf	SpeedMode,W		; Test if SpeedMode already is the minimum allowed value
	andlw	0x03
	subwf	MinSpeedMode,W
	btfsc	STATUS,Z
	return
	decf	SpeedMode,F		; Change to previous allowed speed
	call	DisplayMode
	goto	UpdateRpm
HandleButtonCmd_Button1_SpeedAdj:
	movf	RampDirection,F	; Test if the speed is currently ramping
	btfss	STATUS,Z	
	return					; If it is ramping, then disallow speed adjustments
	movlw	high(MAX_PERIOD); Test if we already ajusted the speed down to the minimum
	subwf	SpeedH,W
	btfsc	STATUS,Z
	return
	movlw	0x1				; Add 1 to speed
	addwf	SpeedL,F
	btfsc	STATUS,C
	incf	SpeedH,F		; Add carry from addition
	movlw	MODE_ADJUST
	movwf	Mode
	call	DisplayMode	
	goto	UpdateAdjustment

HandleButtonCmd_Button2:
	clrf	ButtonCmd
	; Check current mode
	; If MODE_NORMAL 	-> Increment speed setting
	; If MODE_MENU		-> Speed=Speed+0.1
	movf	Mode,F			; Take advantage of the fact, that MODE_NORMAL = 0
	btfss	STATUS,Z
	goto	HandleButtonCmd_Button2_SpeedAdj
HandleButtonCmd_Button2_NextRpm:
 	movf	SpeedMode,W		; Test if SpeedMode already is the maximum allowed value
	andlw	0x03
	subwf	MaxSpeedMode,W
	btfsc	STATUS,Z
	return
	incf	SpeedMode,F		; Change to next allowed speed
	call	DisplayMode
	goto	UpdateRpm
HandleButtonCmd_Button2_SpeedAdj:
	movf	RampDirection,F	; Test if the speed is currently ramping
	btfss	STATUS,Z	
	return					; If it is ramping, then disallow speed adjustments
	movlw	high(MIN_PERIOD); Test if we already ajusted the speed up to the maximum
	subwf	SpeedH,W
	btfsc	STATUS,Z
	return
	movlw	0xFF			; 0xFF = Add -1 to current speed
	addwf	SpeedL,F
	btfss	STATUS,C
	decf	SpeedH,F		; Subtract borrow from the substraction
	movlw	MODE_ADJUST
	movwf	Mode
	call	DisplayMode	
	goto	UpdateAdjustment

HandleButtonCmd_Button3:
	clrf	ButtonCmd
	; Check current mode
	; If MODE_NORMAL 	-> Start/Stop
	; If MODE_MENU		-> Exit from menu
	movf	Mode,F			; Take advantage of the fact, that MODE_NORMAL = 0
	btfss	STATUS,Z
	goto	HandleButtonCmd_Button3_Menu
HandleButtonCmd_Btn3_StartStop:
	btfsc	SpeedMode,RUNNING_BIT
	goto	HandleButtonCmd_Btn3_StartStop2
	movlw	low(MAX_PERIOD)	; Set SpeedL/SpeedH to minimum speed (for ramping)
	movwf	SpeedL			
	movlw	high(MAX_PERIOD)
	movwf	SpeedH		
	call	UpdateRpm		; Read speed to start
	#ifdef SV_ENRICO
	movf	SpeedMode,W		; Read speedmode into WREG
	movwf	SavedSpeedMode
	movlw	1				; = 33 rpm
	movwf	SpeedMode
	movlw	high(65536-40000)	; Set time for special mode
	movwf	SavedSMClockH
	movlw	low(65536-40000)
	movwf	SavedSMClockL
	movlw	1
	movf	SavedSpeedActive
	#endif
	bsf		RampDirection,7	; Fast ramp
	bsf		SpeedMode,RUNNING_BIT	; Start turntable
	goto	DisplayMode
HandleButtonCmd_Btn3_StartStop2:
	bcf		SpeedMode,RUNNING_BIT	; Stop turntable
	goto	DisplayMode
HandleButtonCmd_Button3_Menu:
	clrf	Mode			; Mode = MODE_NORMAL
	movf	EEPROM_Update,F	; Test if the EEPROM must be updated with the adjusted speed
	btfss	STATUS,Z		
	call	UpdateEEPROM_SpeedAdj
	clrf	EEPROM_Update	; Mark as updated.
	goto	DisplayMode	

HandleButtonCmd_Button1Long:
	clrf	ButtonCmd
	; Check current mode
	; If MODE_NORMAL 	-> Set to 33 rpm pulley
	; If MODE_MENU		-> Change selectable speed settings
	movf	Mode,F			; Take advantage of the fact, that MODE_NORMAL = 0
	btfss	STATUS,Z
	goto	HandleButtonCmd_Btn1Long_Menu
HandleButtonCmd_Btn1Long_33P:
	btfsc	SpeedMode,RUNNING_BIT
	return					; Only allow pulley change when stopped
	movlw	MODE_PULLEY33
	movwf	Mode
	call	DisplayMode
	goto	UpdatePulley
HandleButtonCmd_Btn1Long_Menu:
	btfsc	SpeedMode,RUNNING_BIT
	return					; Only allow pulley change when stopped
	movlw	MODE_SPEEDS_DEC
	movwf	Mode
	goto	UpdateAvailableSpeeds

HandleButtonCmd_Button2Long:
	clrf	ButtonCmd
	; Check current mode
	; If MODE_NORMAL 	-> Set to 45 rpm pulley
	; If MODE_MENU		-> Change selectable speed settings
	movf	Mode,F			; Take advantage of the fact, that MODE_NORMAL = 0
	btfss	STATUS,Z
	goto	HandleButtonCmd_Btn2Long_Menu
HandleButtonCmd_Btn2Long_45P:
	btfsc	SpeedMode,RUNNING_BIT
	return					; Only allow pulley change when stopped
	movlw	MODE_PULLEY45
	movwf	Mode
	call	DisplayMode
	goto	UpdatePulley
HandleButtonCmd_Btn2Long_Menu:
	btfsc	SpeedMode,RUNNING_BIT
	return					; Only allow pulley change when stopped
	movlw	MODE_SPEEDS_INC
	movwf	Mode
	goto	UpdateAvailableSpeeds

HandleButtonCmd_Button3Long:
	clrf	ButtonCmd
	; Check current mode
	; If MODE_NORMAL 	-> Menu
	; If MODE_MENU		-> Toggle hi voltage start up time
	movf	Mode,F			; Take advantage of the fact, that MODE_NORMAL = 0
	btfss	STATUS,Z
	goto	HandleButtonCmd_Btn3Long_HiVolt
HandleButtonCmd_Btn3Long_Menu:
	movlw	MODE_MENU
	movwf	Mode
	goto	DisplayMode
HandleButtonCmd_Btn3Long_HiVolt:
	movlw	MODE_VOLTTOGGLE
	movwf	Mode
	goto	VoltToggle


; TestButton
; ----------
; Tests if the key (pointed to by the keymask in WREG) is pressed, released and if it is a short or long press.
TestButton:
	movwf	ButtonTmp		; Store which button is being tested
	movf	PORTA,W			; Read state of buttons from PORTA
	xorwf	ButtonStatus,W	; XOR with current status
	andwf	ButtonTmp,W		; Test if the button has changed state
	btfss	STATUS,Z
	goto	TestButtonUpDown	; Different from current ButtonState
	; Button has not changed - test if it is down
	movf	ButtonStatus,W
	andwf	ButtonTmp,W
	btfss	STATUS,Z
	return					; No, the key is not down
	movf	KeyLongClock,F	; Has key command already been fired?
	btfsc	STATUS,Z
	return					; Yes, a long key command has already been fired
	goto	TestButtonDownClock	; Test if a long keypress has appeared
TestButtonUpDown:
	; Button has changed - up or down?
	movf	PORTA,W
	andwf	ButtonTmp,W
	btfsc	STATUS,Z
	goto	TestButtonDown	; The read key is down
TestButtonUp:
	; Button is currently up - test if it is pending
	movf	ButtonPending,W
	andwf	ButtonTmp,W
	btfss	STATUS,Z		; No, button is not pending - change it to pending
	goto	TestButtonUpClock	; Yes, the button is already pending so test the clock
TestButtonUpCurrentState:
	movf	ButtonStatus,W
	andwf	ButtonTmp,W
	btfss	STATUS,Z
	return					; Button already up
TestButtonUpSetAsPending:
	movf	ButtonTmp,W
	movwf	ButtonPending
	clrf	KeyClock		; Start counter
	clrf	KeyLongClock	; Start long press counter
	return					; Done!
TestButtonUpClock:
	; The button is already pending, so see if it has been debounced
	movlw	BUTTON_DEBOUNCE_COUNT
	subwf	KeyClock,W
	btfss	STATUS,C
	return					; The button has not been debounced yet
TestButtonUpChangeState:
	; The button is up and has been debounced
	; Check if this was a short press and make a short press command if so
	movf	KeyLongClock,F
	btfsc	STATUS,Z
	goto	TestButtonUpClear
	; This was a short press, update the ButtonCmd
	movf	ButtonTmp,W
	movwf	ButtonCmd		; Set button command - short press
TestButtonUpClear:
	movlw	0x07
	movwf	ButtonStatus	; Mark all buttons as up
	clrf	ButtonPending	; No buttons pending
	return
TestButtonDown:
	; Test button is currently down - test if it is pending
	movf	ButtonPending,W
	andwf	ButtonTmp,W
	btfss	STATUS,Z		; No, button is not pending - change it to pending
	goto	TestButtonDownClock	; Yes, the button is already pending so test the clock
TestButtonDownSetAsPending:
	movf	ButtonTmp,W
	movwf	ButtonPending
	clrf	KeyClock		; Start counter
	clrf	KeyLongClock	; Start long press counter
	return					; Done!
TestButtonDownClock:
	; The button is already pending, so see if it has been debounced
	movlw	BUTTON_DEBOUNCE_COUNT
	subwf	KeyClock,W
	btfss	STATUS,C
	return					; The button has not been debounced yet
TestButtonDownChangeState:
	; The button is down and has been debounced
	movlw	BUTTON_LONG_COUNT
	subwf	KeyLongClock,W
	btfsc	STATUS,C
	goto	TestButtonDownLongPress	; Yes - long key press has been detected!
	; No, this is not a long key press (yet)
	incf	KeyLongClock,F	; Increment long key press counter
	clrf	KeyClock		; Start new KeyClock period
	goto	TestButtonExit
TestButtonDownLongPress:
;	btfss	STATUS,Z		; Has exactly BUTTON_LONG_COUNT periods appeared?
;	return					; No, so long keypress message has already been sent!
	incf	KeyLongClock,F	; Increment ButtonLong, so the long keypress message only is set once
	movf	ButtonTmp,W		; The key that generates the long keypress
	iorlw	0x80			; Signal "long keypress"
	movwf	ButtonCmd
	clrf	ButtonPending
	clrf	KeyLongClock
TestButtonExit:
	comf	ButtonTmp,W
	andlw	0x07
	movwf	ButtonStatus
	return


; *****************************
; ***						***
; *** LCD DISPLAY FUNCTIONS ***
; ***						***
; *****************************

; DisplayMode
;
; Displays a text depending on the value in the "Mode" register.
DisplayMode:
	movf	Mode,F			; Test if we are in "Normal" mode
	btfss	STATUS,Z
	goto	DisplayModeMenu
	; We are in "Normal" mode - display current speed.
	call	DisplayMode_Speed
	movlw	High(DisplaySpeedModeJumpTable)
	movwf	PCLATH
	movf	SpeedMode,W	
	andlw	0x03			; Isolate selected speed
	addlw	low(DisplaySpeedModeJumpTable)
	btfsc	STATUS,C
	incf	PCLATH,F
	movwf	PCL	
DisplaySpeedModeJumpTable:
	goto	DisplayMode_16rpm
	goto	DisplayMode_33rpm
	goto	DisplayMode_45rpm
	goto	DisplayMode_78rpm
DisplayModeMenu:
	; We are in a non-"Normal" mode, display the menu text
	movlw	High(DisplayMenuModeJumpTable)
	movwf	PCLATH
	decf	Mode,W			; Disregard "Normal" mode
	addlw	low(DisplayMenuModeJumpTable)
	btfsc	STATUS,C
	incf	PCLATH,F
	movwf	PCL	
DisplayMenuModeJumpTable:
	goto	DisplayMode_Welcome	
	goto	DisplayMode_Menu
	goto	DisplayMode_Pulley33
	goto	DisplayMode_Pulley45
	goto	DisplayMode_VoltToggle
	goto 	DisplayMode_Adjust
	goto	DisplayMode_Speeds
	goto	DisplayMode_Speeds
DisplayMode_Speed:
	LcdHome
	btfss	SpeedMode,RUNNING_BIT
	goto	DisplayMode_Stopped
	movlw	low(sSpeed)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_Stopped:
	LcdHome
	movlw	low(sStopped)-low(sMenuTable)	
	goto	DisplayModeDisplay
DisplayMode_16rpm:
	movlw	low(s16rpm)-low(sMenuTable)	
	call	DisplayModeDisplay
	goto	LcdPad
DisplayMode_33rpm:
	movlw	low(s33rpm)-low(sMenuTable)	
	call	DisplayModeDisplay
	goto	LcdPad
DisplayMode_45rpm:
	movlw	low(s45rpm)-low(sMenuTable)
	call	DisplayModeDisplay
	goto	LcdPad
DisplayMode_78rpm:
	movlw	low(s78rpm)-low(sMenuTable)
	call	DisplayModeDisplay
	goto	LcdPad
DisplayMode_Welcome:
	LcdClear
	movlw	low(sWelcome)-low(sMenuTable)
	call	DisplayModeDisplay
;	call	WaitFor1Sec
	goto	WaitFor1Sec		; Exit through WaitFor1Sec
DisplayMode_Menu:
	LcdClear
	movlw	low(sMenu)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_Pulley33:
	LcdClear
	movlw	low(sPulley33)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_Pulley45:
	LcdClear
	movlw	low(sPulley45)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_VoltToggle:
	LcdClear
	movlw	low(sVoltTime)-low(sMenuTable)
	call	DisplayModeDisplay
	btfsc	HiVolt,1		; Is high voltage enabled?
	goto	DisplayMode_VoltAlways
	movlw	"2"
	btfsc	HiVolt,0		; If bit 0 set then 25 sec else 10 sec
	call	LcdCharOut
	movlw	"1"
	btfss	HiVolt,0		; If bit 0 set then 25 sec else 10 sec
	call	LcdCharOut
	movlw	"5"
	btfsc	HiVolt,0		; If bit 0 set then 25 sec else 10 sec
	call	LcdCharOut
	movlw	"0"
	btfss	HiVolt,0		; If bit 0 set then 25 sec else 10 sec
	call	LcdCharOut
	movlw	low(sSec)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_VoltAlways:
	movlw	low(sAlways)-low(sMenuTable)
	goto	DisplayModeDisplay	
DisplayMode_Speeds:
	LcdClear
	movlw	low(sSpeeds)-low(sMenuTable)
	call	DisplayModeDisplay
	; Current available speeds stored in Speeds:
	; 0 = 16/33/45
	; 1 = 33/45
	; 2 = 33/45/78
	; 3 = All (16/33/45/78)
	movf	Speeds,W
	movwf	SpeedLTmp
	btfsc	STATUS,Z
	goto	DisplayMode_Speeds_16_45
	decf	SpeedLTmp,F
	btfsc	STATUS,Z
	goto	DisplayMode_Speeds_33_45
	decf	SpeedLTmp,F
	btfsc	STATUS,Z
	goto	DisplayMode_Speeds_33_78
DisplayMode_Speeds_All:
	movlw	low(sSpeedsAll)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_Speeds_16_45:
	movlw	low(sSpeeds1645)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_Speeds_33_45:
	movlw	low(sSpeeds3345)-low(sMenuTable)
	goto	DisplayModeDisplay
DisplayMode_Speeds_33_78:
	movlw	low(sSpeeds3378)-low(sMenuTable)
	goto	DisplayModeDisplay		
DisplayMode_Adjust:
	LcdHome
	movlw	low(sAdjust)-low(sMenuTable)
	call	DisplayModeDisplay
	call	GetSpeedNominal
	; Calculate the current offset - SpeedLTmp and SpeedHTmp holds the nominal values
	movf	SpeedL,W
	subwf	SpeedLTmp,F
	btfss	STATUS,C
	decf	SpeedHTmp,F
	movf	SpeedH,W
	subwf	SpeedHTmp,F	
	; Now SpeedLTmp, SpeedHTmp holds the difference
	btfss	SpeedHTmp,7		; Is the number negative?
	goto	DisplayMode_Adjust2
	; Number is negative
	movlw	"-"				; Display a minus
	call	LcdCharOut
	; Negate the number
	comf	SpeedLTmp,F
	comf	SpeedHTmp,F
	movlw	1
	addwf	SpeedLTmp,F
	btfsc	STATUS,C
	incf	SpeedHTmp,F
DisplayMode_Adjust2:
	movlw	0xff
	movwf	DisplayModeTmp	
	; Brute force number to character conversion
	; First, check how many times we can subtract 1000 from the number
DisplayMode_Adjust3:
	incf	DisplayModeTmp,F	; Increment to make the correct character
	movlw	232
	subwf	SpeedLTmp,F
	btfss	STATUS,C
	decf	SpeedHTmp,F
	movlw	3
	subwf	SpeedHTmp,F
	btfss	SpeedHTmp,7		; Test if the number became negative (bit 7 is set)
	goto	DisplayMode_Adjust3	; No, take another loop
	movlw	232
	addwf	SpeedLTmp,F
	btfsc	STATUS,C
	incf	SpeedHTmp,F
	movlw	3
	addwf	SpeedHTmp,F
	movf	DisplayModeTmp,W
	btfss	STATUS,Z		; Do not display if zero
	call	LcdDigitOut		; Display digit
	; Check how many time we can subtract 100 from the number
	movlw	0xff
	movwf	DisplayModeTmp	
DisplayMode_Adjust4:
	incf	DisplayModeTmp,F	; Increment to make the correct character
	movlw	100
	subwf	SpeedLTmp,F
	btfss	STATUS,C
	decf	SpeedHTmp,F
	btfss	SpeedHTmp,7		; Test if the number became negative (by testing bit 7 in SpeedHTmp)
	goto	DisplayMode_Adjust4	; No, take another loop
	movlw	100
	addwf	SpeedLTmp,F
	btfsc	STATUS,C
	incf	SpeedHTmp,F
	movf	DisplayModeTmp,W
	btfss	STATUS,Z		; Do not display if zero
	call	LcdDigitOut		; Display digit
	; Check how many time we can subtract 10 from the number
	movlw	"0"-1
	movwf	DisplayModeTmp	
DisplayMode_Adjust5:
	incf	DisplayModeTmp,F	; Increment to make the correct character
	movlw	10
	subwf	SpeedLTmp,F
	btfsc	STATUS,C
	goto	DisplayMode_Adjust5	; No, take another loop
	movlw	10
	addwf	SpeedLTmp,F
	movf	DisplayModeTmp,W
	call	LcdCharOut		; Display character (always)
	movlw	"."
	call	LcdCharOut
	; Finally, check how many time we can subtract 1 from the number
	movlw	"0"-1
	movwf	DisplayModeTmp	
DisplayMode_Adjust6:
	incf	DisplayModeTmp,F	; Increment to make the correct character
	movlw	1
	subwf	SpeedLTmp,F
	btfsc	STATUS,C
	goto	DisplayMode_Adjust6	; No, take another loop
	movf	DisplayModeTmp,W
	call	LcdCharOut		; Display character
	call	LcdPad
	return
DisplayModeDisplay:
	movwf	DisplayModeTmp	; Store character pointer
DisplayModeDisplay2:
	call	GetMenuTableChar
	iorlw	0
	btfsc	STATUS,Z
	return
	call	LcdCharOut
	incf	DisplayModeTmp,F
	movf	DisplayModeTmp,W
	goto	DisplayModeDisplay2

; Leave out section below - used during original development
#ifdef XYZ
DisplayHex macro
	clrf	STATUS
	movwf	LcdCharTmp
	sublw	9
	movf	LcdCharTmp,W
	btfss	STATUS,C
	addlw	7
	addlw	48
	call	LcdCharOut
	bsf		STATUS,RP1
	bcf		STATUS,RP0
	endm

; For debugging purposes
DisplayEEPromWrite:
	clrf  	STATUS
	LcdClear
	bsf		STATUS,RP1
	swapf	EEADR,W
	andlw	0x0f
	DisplayHex	; Bank = 2
	movf	EEADR,W
	andlw	0x0f
	DisplayHex
	movlw	' '
	clrf	STATUS
	call	LcdCharOut
	bsf		STATUS,RP1
	swapf	EEDATA,W
	andlw	0x0f
	DisplayHex
	movf	EEDATA,W
	andlw	0x0f
	DisplayHex
	call	WaitFor1Sec
	return
#endif

; GetMenuTableChar
; -----------
; Fetch a character from a table and return it in WREG
GetMenuTableChar:
	movwf	MenuCharTmp
	movlw	High(sMenuTable)
	movwf	PCLATH
	movf	MenuCharTmp,W
	addlw	low(sMenuTable)
	btfsc	STATUS,C
	incf	PCLATH,F
	movwf	PCL		
sMenuTable:
sStopped	dt	"Stop: ",0
sSpeed		dt	"Speed: ",0
s16rpm		dt	"16 rpm",0
s33rpm		dt	"33 rpm",0
s45rpm		dt	"45 rpm",0
s78rpm		dt	"78 rpm",0
#ifdef SV_ENRICO
sWelcome	dt	"TTPSU v1.24sv3",0
#else
sWelcome	dt	"TTPSU v1.30",0
#endif
sMenu		dt	"Menu: (-/+/Exit)",0
sPulley33	dt	"Pulley: 33 rpm",0
sPulley45	dt	"Pulley: 45 rpm",0
sVoltTime	dt	"Hi volt: ",0
sAdjust		dt	"Adjust: ",0
sSpeeds		dt	"Speeds: ",0
sSpeeds1645	dt	"16/33/45",0
sSpeeds3345	dt	"33/45",0
sSpeeds3378	dt	"33/45/78",0 
sSpeedsAll	dt	"ALL",0
sSec		dt	" sec.",0
sAlways		dt	"Always",0


; LcdPad
; ------
; Fills the display with spaces from the current position to the end of the line
LcdPad:
	movlw	16
	subwf	LcdPosition,W
	btfsc	STATUS,Z
	return
	movlw	" "
	call	LcdCharOut
	goto	LcdPad

; LcdCtrlOut
;
; Sends a control code to the LCD. 
LcdCtrlOut:
	movwf	LcdCtrlTmp		; Store control code (Wreg)
	andlw	0xf0			; Isolate upper 4 bits of control conde
	movwf	PORTC			; Write to port C
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	bsf		PORTC,LCD_E		; Set E flag
	call	WaitFor1Ms
	bcf		PORTC,LCD_E		; Clear E flag = buffer data
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	swapf	LcdCtrlTmp,W	; Swap to isolate lower 4 bits of control code
	andlw	0xf0			; Isolate upper 4 bits (=lower 4 bits of control code)
	movwf	PORTC			; Write to port C
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	bsf		PORTC,LCD_E		; Set E flag
	call	WaitFor1Ms
	bcf		PORTC,LCD_E		; Clear E flag = buffer data
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	return	

; LcdDigitOut
; -----------
; The WREG as an ASCII character
LcdDigitOut:
	addlw	48
	goto	LcdCharOut

; LcdCharOut
; ----------
; Sends a character to the LCD. 
LcdCharOut:
	movwf	LcdCharTmp		; Store character (Wreg)
	#ifdef	KS0066U
	movlw	8				; Test if we have to change to upper half of display
	subwf	LcdPosition,W
	btfss	STATUS,Z
	goto	LcdCharOut2
	; Shift to upper half of display
	movlw	0xC0			; Set display position to 0x40
	call	LcdCtrlOut
LcdCharOut2:
	movf	LcdCharTmp,W
	#endif
	andlw	0xf0			; Isolate lower 4 bits
	iorlw	0x01			; Set RS bit
	movwf	PORTC			; Write to port C
	nop						; Wait for at least 50 ns. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	bsf		PORTC,LCD_E		; Set E flag
	call	WaitFor1Ms	
	bcf		PORTC,LCD_E		; Clear E flag = buffer data
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	swapf	LcdCharTmp,W	; Upper 4 bits of LcdTmp to lower 4 bits of Wreg
	andlw	0xf0			; Isolate upper 4 bits
	iorlw	0x01			; Set RS bit
	movwf	PORTC			; Write to port C
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	bsf		PORTC,LCD_E		; Set E flag
	call	WaitFor1Ms	
	bcf		PORTC,LCD_E		; Clear E flag = buffer data
	nop						; Wait for at least 50 ms. (nop=200ns @ 20Mhz, 333ns @ 12 MHz and 167 ns. @ 24MHz)
	incf	LcdPosition,F
	return	


; *****************************
; ***						***
; ***    WAIT FUNCTIONS     ***
; ***						***
; *****************************

; WaitForMs
; ---------
; Delays for at least W_REG milliseconds. 
WaitForMs:
	movwf	WaitForTmp
WaitForMsLoop:
	call	WaitFor1Ms
	decfsz	WaitForTmp,F
	goto	WaitForMsLoop
	return


; WaitFor1Ms
; ----------
; Waits for approximately 1 ms.
WaitFor1Ms:
	movlw	high(CLOCK_PERIOD)	; Use high byte of CLOCK_PERIOD
	movwf	WaitForTmp2	
	bcf		STATUS,C	
	rlf		WaitForTmp2,F		; Multiply by 8
	rlf		WaitForTmp2,F
	rlf		WaitForTmp2,F
WaitFor1MsLoop:
	goto	$+1					; Spend 32 cycles
	goto	$+1 				; Actual delay:
	goto	$+1					
	goto	$+1 				
	goto	$+1	
	goto	$+1 
	goto	$+1 
	goto	$+1 
	goto	$+1	
	goto	$+1 
	goto	$+1	
	goto	$+1 
	goto	$+1	
	goto	$+1 
	goto	$+1 
	goto	$+1
	decfsz 	WaitForTmp2,F
	goto	WaitFor1MsLoop
	return

; WaitFor1Sec
; -----------
; Wait for 1 second
WaitFor1Sec:
	movlw	250
	call	WaitForMs
	movlw	250
	call	WaitForMs
	movlw	250
	call	WaitForMs
	movlw	250
	goto	WaitForMs		; Exit through WaitForMs


; ******************************
; ***						 ***
; *** SPEED CHANGE FUNCTIONS ***
; ***						 ***
; ******************************

; UpdateRpm
; -----------
; Updates the timing registers (SpeedL,SpeedH) with the adjusted values of 
; the selected speed (16/33/45/78) and updates the timer with the new period.
UpdateRpm:
	call	ReadEEPROMSpeed	; Read adjusted speed from EEPROM for SpeedMode
	clrf	RampDirection
	movf	SpeedH,W		; Test if we must ramp up or down
	subwf	SpeedHTmp,W
	btfsc	STATUS,Z		
	goto	UpdateRpm2		; SpeedH equal - test SpeedL
	btfss	STATUS,C
	incf	RampDirection,F
	incf	RampDirection,F
	goto	UpdateRpm3
UpdateRpm2:
	movf	SpeedL,W		; Test if we must ramp up or down
	subwf	SpeedLTmp,W
	btfsc	STATUS,Z		
	goto	UpdateRpm3		; SpeedH and SpeedL equal - no ramping
	btfss	STATUS,C
	incf	RampDirection,F
	incf	RampDirection,F
UpdateRpm3:
	movlw	1
	movwf	RampDivider		; Fast ramp - ramp for every interrupt
	call	StartHighVoltage; Enable high voltage during the speed change
	return

; UpdateSpeed
; -----------
; Updates the timing registers (SpeedL,SpeedH) with the adjusted values of 
; the selected speed (16/33/45/78)
UpdateSpeed:
	; Program timer with new values
	bcf		T1CON,TMR1ON	; Temporary stop TIMER1
	movf	SpeedL,W		; Low byte of current speed
	movwf	CCPR1L
	movf	SpeedH,W		; High byte of current speed
	movwf	CCPR1H
	bsf		T1CON,TMR1ON	; Restart TIMER1
	return

; UpdateAdjustment
; ----------------
; Updates the timing registers (SpeedL,SpeedH) and sets the EEPROM_Update 
; flag so the adjustment is written to the EEPROM when exiting the menu.
UpdateAdjustment:
	movlw	1
	movwf	EEPROM_Update	; Signal that the EEPROM must be updated with the adjusted value.
	goto	UpdateSpeed

; UpdatePulley
; ------------
; Updates the pulley setting, resets all speeds with correct timing for the 
; selected pulley and writes the selected pulley to EEPROM. Note: This 
; function are always called when the turntable is stopped! Also, all 
; adjustments are lost and the speed is reset to 33 rpm.
UpdatePulley:
	movlw	ePulley			; Point to ePulley value
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	clrf	STATUS			; Bank 0
	movlw	MODE_PULLEY33	; Check what pulley has been selected
	subwf	Mode,W
	btfsc	STATUS,Z		
	goto	UpdatePulley2	; WREG is zero!
	movlw	12				; 12+33 = 45 rpm
UpdatePulley2:				; WREG is 0 or 12
	addlw	33				; 33 rpm
	call	EEPromWrite		; Write new pulley setting
	clrf	SpeedMode		; 16 rpm
UpdatePulley3:
	call	GetSpeedNominal	; Move nominal speed to SpeedLTmp and SpeedHTmp
	movf	SpeedLTmp,W		; Move nominal speed to SpeedL and SpeedH
	movwf	SpeedL
	movf	SpeedHTmp,W
	movwf	SpeedH
	call	UpdateEEPROM_SpeedAdj	; Update EEPROM with new value
	incf	SpeedMode,F
	btfss	SpeedMode,2		; Has count reached 4 (i.e. has all speeds been updated?)
	goto	UpdatePulley3
	movlw	SPEEDMODE_33RPM ; Reset speed to 33 rpm
	movwf	SpeedMode
	call	UpdateRpm
	call	WaitFor1Sec		; Show "Pulley xxx" messages for 2 seconds
	call	WaitFor1Sec
	movlw	MODE_NORMAL		; Return to normal mode
	movwf	Mode
	goto	DisplayMode

; VoltToggle
; ----------
; Toggle between 10 and 25 seconds start up time.
VoltToggle:
	incf	HiVolt,F		; Increment HiVolt setting
	movlw	3
	subwf	HiVolt,W
	btfsc	STATUS,Z
	clrf	HiVolt
	call	DisplayMode
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movlw	eHiVolt			; Write "eHiVolt" setting
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	clrf	STATUS
	movf	HiVolt,W
	call	EEPromWrite		; Update HiVolt setting in EEPROM
	call	WaitFor1Sec
	call	WaitFor1Sec
	movlw	MODE_MENU
	movwf	Mode
	goto	DisplayMode

; ******************************
; ***						 ***
; ***    EEPROM FUNCTIONS    ***
; ***						 ***
; ******************************

; UpdateEEPROM_SpeedAdj
; ---------------------
; Updates the EEPROM with the adjusted speed (if needed)
UpdateEEPROM_SpeedAdj:
	call	ReadEEPROMSpeed ; Read current adjusted speed from EEPROM (bank=0)
	; Note: EEADR_Tmp is set (by ReadEEPROMSpeed) to point to low byte 
	; of the current speed!
	movf	SpeedL,W
	subwf	SpeedLTmp,F
	btfss	STATUS,Z		; Are the values the same?
	goto	UpdateEEPROM_SpeedAdj2
	; SpeedL and SpeedLTmp are the same, now test SpeedH
	movf	SpeedH,W
	subwf	SpeedHTmp,F
	btfsc	STATUS,Z		; Are the values the same?
	return					; Yes - return without updating the EEPROM
UpdateEEPROM_SpeedAdj2:
	; Adjusted speed is different - update EEPROM with new value
	incf	EEADR_Tmp,W
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	clrf	STATUS			; Bank 0
	movf	SpeedH,W
	call	EEPromWrite		; EEADR already points to the correct byte
	movf	EEADR_Tmp,W
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	clrf	STATUS			; Bank 0
	movf	SpeedL,W
	call	EEPromWrite
	return

; ReadEEPROMSpeed
; ---------------
; Read current adjusted speed from EEPROM - the current speed is
; returned in SpeedLTmp, SpeedHTmp
ReadEEPROMSpeed:
	movf	SpeedMode,W		; Calculate EEPROM address of existing period for this SpeedMode
	movwf	SpeedHTmp		; Store value in SpeedHTmp
	bcf		STATUS,C
	rlf		SpeedHTmp,F		; x2
	movf	SpeedHTmp,W
	addlw	eSpeed16L		; Point to low byte
	movwf	EEADR_Tmp
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	call	EEPromRead		; Read low byte of current speed adjustment (bank=0)
	movwf	SpeedLTmp		; Store in SpeedLTmp
	incf	EEADR_Tmp,W
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movwf	EEADR	
	errorlevel +302			; Enable "Not in bank 0" waring
	call	EEPromRead		; Read high byte of current speed adjustment (bank=0)
	; Compare SpeedL, SpeedH with SpeedLTmp and SpeedHTmp
	movwf	SpeedHTmp
	return 

; EEPromRead
; ----------
; Read a byte from the EEPROM, return in WREG
;	EEADR = Address to read - must be set up in advance
;   Bank set to 0 on exit
EEPromRead:
	bsf		STATUS, RP1		; Select bank 3
	bsf		STATUS, RP0
	errorlevel -302
	bcf		EECON1, WREN	; Clear EEPROM write enable (precausion)
	bcf		EECON1, EEPGD	; Point to data memory (not flash memory)
	bsf		EECON1, RD		; Select EEPROM read
	bcf		STATUS, RP0		; Select bank 2
	movf	EEDATA, W		; Move EEPROM data to W
	errorlevel +302
	clrf	STATUS			; Bank 0
	return
		
; EEPromWrite
; -----------
; Write the byte in WREG to the EEPROM
;	EEADR = Address to write - must be set up in advance
;   Bank set to 0 on exit
EEPromWrite:
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302
	movwf	EEDATA			; Move W (data to write) to EEDATA register
	bsf		STATUS, RP0		; Select bank 3
	bcf		EECON1, EEPGD	; Point to data memory (not flash memory)
	bsf		EECON1, WREN	; Select EEPROM write enable
	bcf		INTCON, GIE		; Disable interrupts
	movlw	0x55			; Start of programming sequence
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf		EECON1, WR		; Perform the write - end of required sequence
	bsf		INTCON, GIE		; Enable interrupts
	errorlevel +302
	clrf	STATUS			; Bank 0
EEPromComplete:
	btfss	PIR2, EEIF		; Test if write has completed
	goto 	EEPromComplete
	bcf		PIR2, EEIF		; Clear write complete flag
	bsf		STATUS, RP1		; Select bank 3
	bsf		STATUS, RP0
	errorlevel -302
	bcf		EECON1, WREN	; Disable EEPROM write enable
;	call	DisplayEEPromWrite	; Debug EEPROM writes (writes EEPROM address and value to display)
	errorlevel +302
	clrf	STATUS			; Bank 0
	return


; *******************************
; ***						  ***
; *** MISCELLANEOUS FUNCTIONS ***
; ***						  ***
; *******************************

; GetSpeedNominal
; ---------------
; Check the EEPROM for current pulley in use, then checks the SpeedMode and stores the
; nominal unadjusted period for the current SpeedMode in SpeedLTmp and SpeedHTmp
GetSpeedNominal:
	movlw	ePulley			; Point to ePulley value
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	call	EEPromRead		; Read the pulley setting (bank=0)
	sublw	33				; Is it the 33 rpm pulley?
	btfss	STATUS,Z
	goto	GetSpeedNominal45
GetSpeedNominal33:
	movf	SpeedMode,W
	andlw	3
	movwf	SpeedLTmp		; Use SpeedLTmp as temporary storage
	btfsc	STATUS,Z		; 16 rpm?
	goto	GetSpeedNominal33_16
	decf	SpeedLTmp,F
	btfsc	STATUS,Z		; 33 rpm?
	goto	GetSpeedNominal33_33
	decf	SpeedLTmp,F
	btfsc	STATUS,Z		; 45 rpm?
	goto	GetSpeedNominal33_45
GetSpeedNominal33_78:
	movlw	166				; 422 = Approximate nominal period for 78 rpm (78.26) at 33 pulley
	movwf	SpeedLTmp
	movlw	1
	movwf	SpeedHTmp
	return
GetSpeedNominal33_45:
	movlw	221				; 733 = Approximate nominal period for 45 rpm at 33 pulley
	movwf	SpeedLTmp
	movlw	2
	movwf	SpeedHTmp
	return
GetSpeedNominal33_33:
	movlw	232				; 1000 = Nominal period for 33 rpm (33 1/3) at 33 pulley
	movwf	SpeedLTmp
	movlw	3
	movwf	SpeedHTmp
	return
GetSpeedNominal33_16:	
	movlw	178				; 1970 = Approximate nominal period for 16 rpm (16.75) at 33 pulley
	movwf	SpeedLTmp
	movlw	7
	movwf	SpeedHTmp
	return
GetSpeedNominal45:
	movf	SpeedMode,W
	andlw	3
	movwf	SpeedLTmp		; Use SpeedLTmp as temporary storage
	btfsc	STATUS,Z		; 16 rpm?
	goto	GetSpeedNominal45_16
	decf	SpeedLTmp,F
	btfsc	STATUS,Z		; 33 rpm?
	goto	GetSpeedNominal45_33
	decf	SpeedLTmp,F
	btfsc	STATUS,Z		; 45 rpm?
	goto	GetSpeedNominal45_45
GetSpeedNominal45_78:
	movlw	63				;  575 = Approximate nominal period for 78 rpm (78.26) at 45 pulley
	movwf	SpeedLTmp
	movlw	2
	movwf	SpeedHTmp
	return
GetSpeedNominal45_45:
	movlw	232				; 1000 = Nominal period for 45 rpm at 45 pulley
	movwf	SpeedLTmp
	movlw	3
	movwf	SpeedHTmp
	return
GetSpeedNominal45_33:
	movlw	70				; 1350 = Nominal period for 33 rpm (33 1/3) at 45 pulley
	movwf	SpeedLTmp
	movlw	5
	movwf	SpeedHTmp
	return
GetSpeedNominal45_16:	
	movlw	127				; 2667 = Approximate nominal period for 16 rpm (16.75) at 45 pulley
	movwf	SpeedLTmp
	movlw	10
	movwf	SpeedHTmp
	return

; UpdateAvailableSpeeds
; ---------------------
; Updates the number of available speeds.
UpdateAvailableSpeeds:
	movlw	MODE_SPEEDS_INC
	subwf	Mode,W
	btfss	STATUS,Z
	goto	UpdateAvailableSpeedsDec
UpdateAvailableSpeedsInc:
	incf	Speeds,F
	btfss	Speeds,2		; Speeds > 3?
	goto	UpdateAvailableSpeedsDisplay
	decf	Speeds,F
	goto	UpdateAvailableSpeedsDisplay
UpdateAvailableSpeedsDec:
	movf	Speeds,F
	btfsc	STATUS,Z
	goto	UpdateAvailableSpeedsDisplay
	decf	Speeds,F
UpdateAvailableSpeedsDisplay:
	; Write to EEPROM
	bsf		STATUS, RP1		; Select bank 2
	bcf		STATUS, RP0
	errorlevel -302			; Enable "Not in bank 0" waring
	movlw	eSpeeds			; Write "eSpeeds" setting
	movwf	EEADR
	errorlevel +302			; Enable "Not in bank 0" waring
	clrf	STATUS
	movf	Speeds,W		; Write Speeds setting
	call	EEPromWrite		
	; Update speeds and display
	call	UpdateSpeeds
	movlw	SPEEDMODE_33RPM
	movwf	SpeedMode
	call	UpdateRpm
	goto	DisplayMode

; UpdateSpeeds
; ------------
; Sets MinSpeedMode and MaxSpeedMode based on the setting of the
; Speeds variable.
UpdateSpeeds:
	movf	Speeds,W
	movwf	SpeedLTmp
	btfsc	STATUS,Z
	goto	UpdateSpeeds_16_45
	decf	SpeedLTmp,F
	btfsc	STATUS,Z
	goto	UpdateSpeeds_33_45
	decf	SpeedLTmp,F
	btfsc	STATUS,Z
	goto	UpdateSpeeds_33_78
UpdateSpeeds_All:
	clrf	MinSpeedMode	; 16 rpm
	movlw	3
	movwf	MaxSpeedMode	; 78 rpm
	return
UpdateSpeeds_16_45:
	clrf	MinSpeedMode	; 16 rpm
	movlw	2
	movwf	MaxSpeedMode	; 45 rpm
	return
UpdateSpeeds_33_45:
	movlw	1
	movwf	MinSpeedMode	; 33 rpm
	movlw	2
	movwf	MaxSpeedMode	; 45 rpm
	return
UpdateSpeeds_33_78:
	movlw	1
	movwf	MinSpeedMode	; 33 rpm
	movlw	3
	movwf	MaxSpeedMode	; 78 rpm
	return

; StartHighVoltage
; ----------------
; Configures PORTA<3> as output, set it low and set up HiVoltCounter.
StartHighVoltage:
	bsf		STATUS,RP0		; Bank 1
	errorlevel -302			; Do not report "Not in Bank 0" warning
	movlw	0xF7
	movwf	TRISA			; PORTA<0:2,4:7> as inputs -  <3> as output.
	errorlevel +302
	clrf	STATUS			; Bank 0
	bcf		PORTA,VOLTAGE_CTRL	; Set PORTA<3> low
	; Set up HiVoltCOunter
	btfsc	HiVolt,0		; Test how long the high voltage must be maintained.
	goto	HiVolt25
HiVolt10:
	movlw	high(65536-10000)
	movwf	HiVoltClockH
	movlw	low(65536-10000)
	movwf	HiVoltClockL
	return	
HiVolt25:
	movlw	high(65536-25000)
	movwf	HiVoltClockH
	movlw	low(65536-25000)
	movwf	HiVoltClockL
	return	

; ******************************
; ***						 ***
; ***        RAM AREA        ***
; ***						 ***
; ******************************

; RAM 
; ---
; RAM area. On PIC16F870 the special function registers
; ranges from 0x00 to 0x1F. The bytes at 0xF0..0xFF, 0x170..0x17F
; and 0x1F0..0x1FF all access bank 0 0x70..0x7F.
RAM:
	cblock	0x20		; Bank 0 registers
		DacPeriod		; The current DAC period in use. Counts down to zero.
		DacValue		; The _next_ DAC value to be outputted
		DacDirection	; Increment or decrement the DAC value? Holds value 1 or 255.
		DacNextPeriod	; The _next_ DAC period to use.
		; SpeedL and SpeedH is the speed the SpeedBox is currently
		; running at. If using the 45rpm pulley and running at
		; 45 rpm speed, the nominal value is 1000. 1001 incicates
		; a -0.1% speed adjustment, 999 a + 0.1% speed adjustment.
		; A value of 1350 will run the turntable at 33.33 rpm of 
		; the 45 rpm pulley.
		; NOTE: The actual values are corrected for the clock
		;		cycles it actually takes for the 
		;		InterruptServiceRoutine to reprogram TIMER1,
		;		which is defined by ISR_TIMER1_DELAY.
		SpeedL			; SpeedL/SpeedH holds the current speed, ie. 
		SpeedH			; 1000 = 100.0%.
		SpeedMode		; Current speed mode (0..3=16/33/45/78 rpm, bit 7: 0=Stopped, 1=Running)
		Speeds			; The available speeds (see eSpeeds in EEPROM area)
		MinSpeedMode	; Minimum value for SpeedMode (from setup, 1..2, 0..2, 1..3 or 0..3)
		MaxSpeedMode	; Maximum value for SpeedMode (see above)
		; Mode determines the menu mode the TTPSU is in:
		; 0x00: Normal (Display speed)
		; 0x01: Welcome
		; 0x02:	Menu
		Mode			
		; The "KeyClock" counts 1 up every approx. 1 ms. The counter works this way: 
		; ClockCounterL and ClockCounterH holds the number of cycles needed before 
		; the next increment of KeyCount must occur. For every interrupt, 
		; SpeedL/SpeedH is subtracted from ClockCounterL/H and when the result becomes 
		; negative, KeyClock are incremented and CLOCK_PERIOD is added to the 
		; ClockCounterL/H register pair. This way the KeyClock will average an 1 ms. 
		; period over time. The value for ClockCounterL/H is dependant on the
		; X-tal speed - see the CLOCK_PERIOD constant for timing.
		KeyClock		; Internal clock for keyboard timing.
		KeyLongClock	; Counter to see if a button has been hold down long enough to quaify for a long press.
		ClockCounterL	; Hold the number of Tcy's needed, before KeyClock
		ClockCounterH	; is updated.
		WaitForTmp		; Temporary variable used by the WaitForMs routine.
		WaitForTmp2		; Temporary variable used by the WaitFor1Ms routine.
		ButtonStatus	; Last debounced status of buttons.
		ButtonPending	; Stores which button that currently are being debounced.
		ButtonCmd		; Bitmask with the button that has been pressed (if bit 7 set, then a long press!)
		EEPROM_Update	; When non-zero, the speed adjustment will be updated when exiting the menu
		SpeedLTmp		; Temporary register for SpeedL
		SpeedHTmp		; Temporary register for SpeedH
		HiVolt			; Bit 0: 0 = 10 seconds, 1 = 25 seconds (Approximate times)
						; Bit 1: 0 = use time above, 1 = Always high voltage.
		HiVoltClockL	; Counter for high voltage time (in milliseconds).
		HiVoltClockH
		EEADR_Tmp		; Temporary register for EEADR
		RampDirection	; 1=Ramp up, 2=Ramp down, 0=No ramp - speed has been reached. Bit 7 set for fast ramp.
		RampDivider		; Pause between changing the ramp-valuea
		; Below is only used when SV_ENRICO is defined.
		SavedSpeedActive; Special speed mode active when SV_ENRICO is defined.
		SavedSpeedMode	; Hold final speed mode when SV_ENRICO is defined.
		SavedSMClockL	; Time counter for SV_ENRICO mode
		SavedSMClockH
	endc
	; 0x70 kept free to allow debugger to be used.
	cblock	0x71		; Shared registers - can be accessed from all banks.
		W_Temp			; Saves W register during interrupt.
		STATUS_Temp		; Saves STATUS register during interrupt.
		PCLATH_Temp		; Saves PCLATH during interrupts (when needed)
		LcdCharTmp		; Temporary register used by LcdCharOut function
		LcdCtrlTmp		; Temporary register used by LcdCtrlOut function
		LcdPosition		; Next position on LCD where to output character
		ButtonTmp		; Temporary register for testing button state
		DisplayModeTmp	; Temporary register for DisplayMode function
		MenuCharTmp		; Temporary register for GetMenuChar function
                DebugTemp
	endc

; ******************************
; ***						 ***
; ***        RAM AREA        ***
; ***						 ***
; ******************************

; EEPROM 
; ------
; Addresses of values stored in EEPROM
ePulley		equ		0x00	; Address of selected Pulley (Possible values: 33 or 45, default = 45)
eSpeeds		equ		0x01	; The available speeds (0=16/33/45, 1=33/45, 2=33/45/78, 3=16/33/45/78)
eHiVolt		equ		0x02	; Current start up time with high voltage output.
eSpeed16L	equ		0x10	; Adjusted period for 16 rpm
eSpeed16H	equ		0x11
eSpeed33L	equ		0x12	; Adjusted period for 33 rpm
eSpeed33H	equ		0x13
eSpeed45L	equ		0x14	; Adjusted period for 45 rpm
eSpeed45H	equ		0x15
eSpeed78L	equ		0x16	; Adjusted period for 78 rpm
eSpeed78H	equ		0x17

; Initial values for EEPROM
		org		0x2100
		de		45		; Pulley
		de		1		; Speeds
		de		0		; Hi voltage start up time. Default = 0 (Approx. 5 sec.)
		org		0x2110
		de		127,10	; 2667 = Approximate nominal period for 16 rpm (actually 16.75) at 45 pulley
		de		70,5	; 1350 = Nominal period for 33 rpm (actually 33 1/3) at 45 pulley
		de		232,3	; 1000 = Nominal period for 45 rpm at 45 pulley
		de		63,2	;  575 = Approximate nominal period for 78 rpm (actually 78.26) at 45 pulley
	end
 